<?php 

/**
 * Objecto que representa um período com início e fim no contexto do horário de um dia.
 *
 */

class HorarioSemana {

    var $id;

    // Referência para o objecto HorarioLoja detentor
    var $horario;
    // Um unix timestamp que represente a data de início da semana (se for de excepção)
    // ou null se for a semana geral.
    var $data;

    // Has Many associations
    var $periodos = array();
    var $periodos_por_dia = array(); // will contain other arrays

    /*
     * Default Constructor
     */
    function __construct($id = null)
    {
        $this->id = $id;
    }

    function getId()
    {
        return $this->id;
    }
    function setId($id)
    {
        $this->id = $id;
    }

    function setHorario($horario)
    {
        $this->horario = $horario;
    }

    /**
     * Definir a data (timestamp) em que começa esta semana se for uma
     * semana de excepção ou null se for a semana base.
     * @param timestamp $data
     */
    function setData($data)
    {
        $this->data = $data;
    }
    function getData()
    {
        return $this->data;
    }

    function addPeriodo($periodo) {
        $per =& $periodo;
        if (empty($this->periodos_por_dia[$periodo->dia])) {
            $this->periodos_por_dia[$periodo->dia] = array();
        }
        $this->periodos_por_dia[$per->dia][] = $per;
        $this->periodos[] = $per;
    }

    /**
     *
     * @param WeekDay $day
     * @return array an array with all the periods of the given day
     */
    function getPeriodos($day) {
        return @$this->periodos_por_dia[$day];
    }

    /**
     *
     * @param <type> $day
     * @param string $time hh:mm such as "0:12" or "23:30"
     * @return
     */
    function isWorkTime($day, $time, $pause_is_work = false) {
        if (!is_array($this->periodos_por_dia[$day])) {
            return false;
        }

        foreach ($this->periodos_por_dia[$day] as $periodo) {


	    if ($pause_is_work) {
		if ($periodo->type != Periodo::WORK && $periodo->type != Periodo::PAUSE) {
		    continue;
		}
	    }
	    else {
		if ($periodo->type != Periodo::WORK) {
		    continue;
		}
	    }

            if ($periodo->isTimeDuringPeriod($time) == true) {
                return true;
            }
            // check the other periods
        }

        return false;
    }

    /**
     *
     * @return string the year and week number in the format "YYYY-WW" or null if no date is specified
     */
    function getYearWeek() {
        if ($this->data == null)
            return null;
        return date("Y-W", $this->data);
    }

	function getEarliestStartingPeriodToday() {
		$day = WeekDay::today();
		return $this->getEarliestStartingPeriod($day);
	}

	function getEarliestStartingPeriod($day) {
		$periodos = $this->getPeriodos($day);

		if ($periodos == null) {
			return null;
		}

		$earliestPeriodo = null;
		$earliestStart = PHP_INT_MAX;

		foreach ($periodos as $periodo) {
			$periodoStart = $periodo->getInicioInMinutes();
			if ($periodoStart < $earliestStart) {
				$earliestPeriodo = $periodo;
				$earliestStart = $periodoStart;
			}
		}
		
		return $earliestPeriodo;
	}

	function getLatestEndingPeriodToday() {
		$day = WeekDay::today();
		return $this->getLatestEndingPeriod($day);
	}

	function getLatestEndingPeriod($day) {
		$periodos = $this->getPeriodos($day);

		if ($periodos == null) {
			return null;
		}

		$latestPeriodo = null;
		$latestEnd = (PHP_INT_MAX * -1); // more or less min int value

		foreach ($periodos as $periodo) {
			$periodoEnd = $periodo->getFimInMinutes();

			if ($periodoEnd > $latestEnd) {
				$latestPeriodo = $periodo;
				$latestEnd = $periodo->getFimInMinutes();
			}
		}
		return $latestPeriodo;
	}


	/**
	 * Checks if the service was continuosly open
	 * between yesterday and today (no break at midnight).
	 *
	 * @return true if it was continuosly open, false otherwise
	 */
	function isContinuouslyOpenToday() {
		$timestampNow = time();
		$today = WeekDay::today();

		$openingPeriodoToday = $this->getEarliestStartingPeriod($today);

		if ($openingPeriodoToday == null || $openingPeriodoToday->getInicio() != "0:00" && $openingPeriodoToday->isDiaInteiro() == false) {
			// Today's opening period didn't start at the very
			// beginning of the day and as such today is not
			// a continuosly open day.
			return false;
		}

		if ($today == WeekDay::MONDAY) {
			// Today is Monday which means that yesterda, Sunday,
			// belonged to the previous week.
			// Eventually it will be exactly the same week... but it might not be.
			$timestampYesterday = strtotime("yesterday", $timestampNow);
			$closingWeek = $this->horario->getHorarioSemanaByTimestamp($timestampYesterday);
			$yesterday = WeekDay::SUNDAY;
		}
		else {
			$yesterday = $today - 1;
			$closingWeek = $this;
		}

		$closingPeriodoYesterday = $closingWeek->getLatestEndingPeriod($yesterday);

		if ($closingPeriodoYesterday instanceof Periodo &&
			($closingPeriodoYesterday->isDiaInteiro()
			|| $closingPeriodoYesterday->getFim() == "24:00")) {
			return true;
		}
		return false;
	}













    function validate() {
        if ($this->horario == null || ($this->horario->getId() > 0) == false)
        {
            return false;
        }
        return true;
    }

    /**
     *
     * @param AccessoDB $bd a database connection object
     * @param boolean $validate whether to validate this Periodo object before saving it
     * @return boolean whether the operation was successful or not
     */
    function save($bd, $validate = true) {

        if ($validate == true) {
            if ($this->validate() == false) {
                debug("HorarioSemana ".$this->id." didn't validate.");
                return false;
            }
        }

        $oldPeriodosIds = array();

        if ($this->id > 0) {
            // Update an existing HorarioSemana

            $sql = "UPDATE horario_semana SET ";
            $sql.= "id_horario = ".$this->horario->getId().", ";
            if (empty($this->data)) {
                $sql.= "start_date = NULL ";
            }
            else {
                $sql.= "start_date = ".date("Y-m-d H:i:s Z", $this->data)." ";
            }
            $sql.= "WHERE id = ".$this->id.";";

            // We must also check which Periodo objects exist associated
            // to this HorarioSemana because we will save the Periodo objects
            // below and need to know if there are periods that have to be
            // removed.
            $sql = "SELECT id FROM periodo WHERE id_horario_semana = ".$this->id.";";
            $result = $bd->executarSQL($sql);
            if ($result == false){
                // Some error occurred
                return false;
            }

            while (($row = $bd->obterRegisto($result)) !== false) {
                $oldPeriodosIds[] = $row['id'];
            }
        }
        else {
            $sql = "INSERT INTO horario_semana(id, id_horario, start_date) VALUES(";
            $sql.= "0, ";
            $sql.= $this->horario->getId().", ";
            if (empty($this->data)) {
                $sql.= "NULL); ";
            }
            else {
                $sql.= "'".date("Y-m-d H:i:s", $this->data)."'); ";
            }

            if ($bd->executarSQL($sql) == false) {
                // Some error occurred
                return false;
            }
            // Get the autoincremented id
            $this->id = $bd->obterUltimoInsertId();
        }

        // Cascade the delete/update to the Periodo objects
        foreach ($this->periodos as $periodo) {
            if ($periodo->save($bd) == false) {
                return false;
            }
        }

        // Get the ids of all the Periodo objects that we have for this week
        $newPeriodosIds = array();
        foreach ($this->periodos as $periodo) {
            $newPeriodosIds[] = $periodo->id;
        }

        // Delete the old periodo objects which
        // do not belong to this week anymore
        $periodoIdsToDelete = array_diff($oldPeriodosIds, $newPeriodosIds);

        if (count($periodoIdsToDelete) > 0) {
            $sql = "DELETE FROM periodo WHERE ";
            for($i = count($periodoIdsToDelete)-1; $i > 0; $i--)
                $sql .= "id = " . $periodoIdsToDelete[$i] . " OR ";
            $sql .= "id = " . $periodoIdsToDelete[0] . ";";
            if ($bd->executarSQL($sql) == false) {
                // Some error occurred
                return false;
            }
        }

        return true;
    }

    
}
?>