<?php 

/**
 * Objecto que representa um período com início e fim no contexto do horário de um dia.
 *
 */

class Periodo {

    // The constants to use as the Period type
    const WORK = 0;
    const PAUSE = 1;    // couldn't use the more correct word "BREAK" as it is reserved
    const MEAL = 2;

    var $id;

    // Wheter this period encompasses the 24h of the day (for 24/7 days). 
    // In this case there shall be no $inicio and $fim
    var $diaInteiro;
    var $inicio;     // string do tipo "14:15"
    var $fim;        // string do tipo "14:15"
    var $dia;        // HorarioSemana::MONDAY, HorarioSemana::TUESDAY, ...
    var $tipo;       // Whether it is a working period, a break, etc.

    var $horarioSemana;

    /*
     * Default Constructor
     */
    function __construct($id = null, $diaInteiro = null, $inicio = null, $fim = null, $dia = null, $tipo = null, $horarioSemana = null)
    {
        $this->id = $id;

        $this->diaInteiro = $diaInteiro;
        $this->inicio = $inicio;
        $this->fim = $fim;
        $this->dia = $dia;
        $this->tipo = $tipo;
        $this->horarioSemana = $horarioSemana;
    }

    function getId(){
        return $this->id;
    }
    function setId($id){
        $this->id = $id;
    }

    function isDiaInteiro(){
	if ($this->diaInteiro ||
		$this->inicio == "0:00" &&
		$this->fim == "24:00") {
		return true;
	}
	return false;
    }
    function setDiaInteiro($value){
        $this->diaInteiro = $value;
    }

    /**
     * @return int the number of minutes passed since 0:00 up to the "inicio" time
     */
    function getInicioInMinutes() {
        if ($this->diaInteiro) {
           return 0;
        }
        return $this->timeStringToMinutes($this->inicio);
    }
    function getInicio(){
        return $this->inicio;
    }
    function setInicio($value){
        $this->inicio = $value;
    }

    /**
     * @return int the number of minutes passed since 0:00 up to the "fim" time
     */
    function getFimInMinutes() {
        if ($this->diaInteiro) {
           return 1440; // 24h * 60 min
        }
        return $this->timeStringToMinutes($this->fim);
    }
    function getFim(){
        return $this->fim;
    }
    function setFim($value){
        $this->fim = $value;
    }

    function getDia(){
        return $this->dia;
    }
    function setDia($value){
        $this->dia = $value;
    }

    function getTipo(){
        return $this->tipo;
    }
    function setTipo($value){
        $this->tipo = $value;
    }

    function getHorarioSemana(){
        return $this->horarioSemana;
    }
    function setHorarioSemana($value){
        $this->horarioSemana = $value;
    }

    /**
     * Checks if a given time is inside this period. A time is inside this period
     * if it is equal or later than the start time and if it ends before the end time.
     *
     * @param string $time a time such as "21:20"
     * @return boolean true if the time is inside this period, false otherwise
     */
    function isTimeDuringPeriod($time) {
        $time_minutes = $this->timeStringToMinutes($time);

        return $time_minutes >= $this->getInicioInMinutes() &&
                $time_minutes < $this->getFimInMinutes();
    }

    /**
     *
     * @param string $time something like "0:22" or "13:01"
     * @return integer the number of minutes passed since midnight up to $time or -1 if the number is invalid.
     */
    private function timeStringToMinutes($time) {
        $elements = explode(":", $time);
        if ($elements < 2)
            return -1; // Not a valid timestring

        if ($elements[0] >= 0 && $elements[0] <= 23
                && $elements[1] >= 0 && $elements[1] <= 59) {

            $minutes = $elements[0] * 60; // hours * 60
            $minutes+= $elements[1];      // + minutes
            return $minutes;
        }
	else if ($elements[0] == 24 && $elements[1] == 0) {
	    // "24:00" time which is basically a full day
	    return 1440; // 24h * 60 min
	}

        // Not a valid time string
        return -1;
    }

    /*
     * Checks if all the data in this Periodo is valid.
     * @return boolean true if the Periodo has only valid data, false otherwise
     */
    function validate() {

        if ($this->diaInteiro) {
            // Full day period
            $this->diaInteiro = true; // make sure the value it has is an actual boolean
            // if it is a full day, we don't need inicio and fim defined
            $this->inicio = "0:00";
            $this->fim = "24:00";
        }
        else {
            // Not a full day period
            $this->diaInteiro = false; // make sure it is a boolean false
            $startInMinutes = $this->timeStringToMinutes($this->inicio);
            $endInMinutes = $this->timeStringToMinutes($this->fim);
            // Check if the start and end times are valid -> "hh:mm" and that the end is after the start
            if ($startInMinutes == -1 || $endInMinutes == -1 || $endInMinutes < $startInMinutes) {
                return false;
            }

        }

        // Check if $dia is valid
        $days = array(WeekDay::MONDAY,
            WeekDay::TUESDAY, WeekDay::WEDNESDAY, WeekDay::THURSDAY,
            WeekDay::FRIDAY, WeekDay::SATURDAY, WeekDay::SUNDAY );
        if (in_array($this->dia, $days) == false) {
            return false;
        }

        // Check if $tipo is valid
        $periodTypes = array(Periodo::WORK, Periodo::PAUSE, Periodo::MEAL);
        if (in_array($this->tipo, $periodTypes) == false) {
            return false;
        }

        if (($this->horarioSemana instanceof HorarioSemana) == false ||
                ($this->horarioSemana->getId() > 0) == false ) {
            // $horarioSemana is either not a HorarioSemana object or
            // has not an id yet, and we need that id!
            return false;
        }
        
        return true;

    }

    /**
     *
     * @param AccessoDB $bd a database connection object
     * @param boolean $validate whether to validate this Periodo object before saving it
     * @return boolean whether the operation was successful or not
     */
    function save($bd, $validate = true) {

        if ($validate == true) {
            if ($this->validate() == false) {
		debug("Periodo ".$this->id." didn't validate.");
                return false;
            }
        }

        $diaInteiro = $this->diaInteiro ? 1 : 0;

        if ($this->id > 0) {
            // We're updating an existing Periodo

            $sql = "UPDATE periodo SET ";
            $sql.= "inicio = '".$this->inicio."', ";
            $sql.= "fim = '".$this->fim."', ";
            $sql.= "dia_inteiro = ".$diaInteiro.", ";
            $sql.= "tipo = ".$this->tipo.", ";
            $sql.= "dia = ".$this->dia.", ";
            $sql.= "id_horario_semana = ".$this->horarioSemana->id." ";
            $sql.= "WHERE id = ".$this->id.";";
            if ($bd->executarSQL($sql) == false) {
                return false;
            }
        }
        else {
            $sql = "INSERT INTO periodo (dia_inteiro, inicio, fim, tipo, dia, id_horario_semana) VALUES(";
            $sql.= $diaInteiro.", ";
            $sql.= "'".$this->inicio."', ";
            $sql.= "'".$this->fim."', ";
            $sql.= $this->tipo.", ";
            $sql.= $this->dia.", ";
            $sql.= $this->horarioSemana->id.");";
            if ($bd->executarSQL($sql) == false) {
                // Some error occurred
                return false;
            }
            // Get the autoincremented id
            $this->id = $bd->obterUltimoInsertId();
        }

        return true;
    }


}
?>