<?php  

class ScheduleController extends ControllerBase {

	// Components can be accessed as $this->Component ("ComponentName" => "$this->ComponentName")
	var $components = array();
	// Helpers are located in 'views/helpers'. They will be loaded and provided to the view. Example: "Xml" => $xml
	var $helpers = array();

	function ScheduleController() {
		parent::__construct();
	}


	/**
	 * book a scheduled seat
	 * @param service_id: service id
	 * @param datetime: timestamp of the wanted seat
	 * @param schedule_service_id: optional schedule service id
	 * @param location_id: optional location associated with the service the client wants
	 * @param client_name: optional argument with the name of the client, to a better identification
	 * @param comments: optional argument with some helpfull comments for the client
	 **/
	function issueTicket(){
		$service_id = $this->request->params['service_id'];
		$datetime = $this->request->params['datetime'];

		$servicesObjModel = null;
		// schedule service id is optional, if not specified we will pick the first one from the system
		if(!isset($this->request->params['schedule_service_id']) || $this->request->params['schedule_service_id'] == ''){
			// get the first one
			if($servicesObjModel == null){
				require_once("services/MethodBase.class.php");
				require_once("services/contactline/services.class.php");
				$servicesObjModel = new Services($this->request);
			}
			$servicesResultSet = $servicesObjModel->getServices(array(), "agendamentos");

			if(count($servicesResultSet) == 0){
				$this->set('success', "FALSE");
				$this->set('message', "There is no schedule service in the system!");
				return;
			}
			$schedule_service_id = $servicesResultSet[0]['id_servico'];

		} else {
			$schedule_service_id = $this->request->params['schedule_service_id'];
		}
		

		// location id is "optional", if not specified get the first one from destination service
		if(!isset($this->request->params['location_id']) || $this->request->params['location_id'] == ''){
			// get the first one
			if($servicesObjModel == null){
				require_once("services/MethodBase.class.php");
				require_once("services/contactline/services.class.php");
				$servicesObjModel = new Services($this->request);
			}
			$servicesResultSet = $servicesObjModel->getServices(array($service_id));
			
			if(count($servicesResultSet) == 0){
				$this->set('success', "FALSE");
				$this->set('message', "There is no service identified by '$service_id'!");
				return;
			}
			$locations = $servicesObjModel->getServiceLocations($servicesResultSet[0]['id_servico'], $serviceResultSet[0]['id_entidade']);
			if(count($locations) == 0){
				$this->set('success', "FALSE");
				$this->set('message', "There is no location associated with service identified by '$service_id'!");
				return;
			}
			$location_id = $locations[0]['id_loja'];

		} else {
			$location_id = $this->request->params['location_id'];
		}

		// optional parameters
		$client_name = "";
		if(isset($this->request->params['client_name'])){
			$client_name = urldecode($this->request->params['client_name']);
		}
		$comments = "";
		if(isset($this->request->params['comments'])){
			$comments = urldecode($this->request->params['comments']);
		}

		// check if datetime is valid
		// > time()
		if($datetime < time()){
			$this->set('success', "FALSE");
			$this->set('message', "Datetime argument must be greater than 'next_available_seat'!");
			return;
		}
		
		// > next_available_seat
		if($servicesObjModel == null){
			require_once("services/MethodBase.class.php");
			require_once("services/contactline/services.class.php");
			$servicesObjModel = new Services($this->request);
		}
		$servicesResultSet = $servicesObjModel->getServices(array($service_id));
		if(count($servicesResultSet) == 0){
			$this->set('success', "FALSE");
			$this->set('message', "There is no service identified by '$service_id'!");
			return;
		}
		// get queue size 
		$queue_size = $servicesObjModel->getQueueSize($service_id, $location_id);

		// get info about average_waiting_time 
		if(isset($servicesResultSet[0]['previsao_atendimento']) && $servicesResultSet[0]['previsao_atendimento'] != ''){
			$average_waiting_time = $servicesResultSet[0]['previsao_atendimento'];
		} else {
			$average_waiting_time = $servicesObjModel->getAverageWaitingTime($service_id, $location_id);
		}

		// calculate what should be the next available seat
		$timeParts = explode(":", $average_waiting_time);
		$waiting_time_seconds = ($timeParts[0] * 60 * 60) + ($timeParts[1] * 60) + $timeParts[2];
		$next_available_seat_seconds = $service['queue_size'] * $waiting_time_seconds;
		$next_available_seat = time() + $next_available_seat_seconds;

		if($datetime < $next_available_seat){
			$this->set('success', "FALSE");
			$this->set('message', "Datetime argument must be greater than 'next_available_seat'!");
			return;
		}


		// ISSUE the TICKET
		require_once("services/MethodBase.class.php");
		require_once("services/contactline/tickets.class.php");
		$ticketsObjModel = new Tickets($this->request);

		// generate ticket code
		while(true){
			$ticket_code = str_pad(mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);

			// check if is not yet used
			$ticketsResultSet = $ticketsObjModel->getTicketByCode($ticket_code);
			if(count($ticketsResultSet) > 0) {
				// check if any ticket is still valid
				$tickets_valid = 0;
				foreach($ticketsResultSet AS $ticket){
					if($ticket['senhas_status'] == SENHA_AGENDAMENTO){
						$tickets_valid++;
						break;
					}
				}
				if($tickets_valid == 0){
					break;
				}
			} else {
				// no tickets with this code
				break;
			}
		}

		// insert into DB
		$ticket = array(
			"numero_senha" => $ticket_code,
			"id_loja" => $location_id,
			"servico_emissao" => $schedule_service_id,
			"servico_destino" => $service_id,
			"status" => SENHA_AGENDAMENTO,
			"observacoes" => $comments,
			"nome_cliente" => $client_name,
			"scheduled" => date("Y-m-d H:i:s", $datetime),
		);
		$newID = $ticketsObjModel->createTicket($ticket);

		if($newID == null){
			// error
			$this->set('success', "FALSE");
			$this->set('message', "Error creating scheduled ticket!");
		}

		$ticketsResultSet = $ticketsObjModel->getTicketByCode($ticket_code);
		if(count($ticketsResultSet) > 0){
			$tickets = array("ticket" => array());
			$tickets['ticket'][] = $this->getTicketFromResultSet($ticketsResultSet[0], $ticketsObjModel);
			$data = array('tickets' => $tickets);

			$this->set('data', $data);

		} else {
			$this->set('success', "FALSE");
			$this->set('message', "ticket code not recognized!");
		}
	}

	/**
	 * validates a schedule ticket type (inside some constraints)
	 * @param ticket_code: ticket_number of the scheduled ticket
	 **/
	function validateTicket(){
		$ticket_code = $this->request->params['ticket_code'];
		
		require_once("services/MethodBase.class.php");
		require_once("services/contactline/tickets.class.php");
		$ticketsObjModel = new Tickets($this->request);
		$ticketsResultSet = $ticketsObjModel->getTicketByCode($ticket_code);
		
		if(count($ticketsResultSet) > 0){
			$ticket = $ticketsResultSet[0];

			// verify if is still valid
			if($ticket['senhas_status'] != SENHA_AGENDAMENTO){
				$this->set('success', "FALSE");
				$this->set('message', "Ticket code not valid anymore!");

			} else {
				// verify time constraints
				$ticketScheduleTime = $ticket['senhas_scheduled_timestamp'];
				$lowerLimit = (time() - (SCHEDULE_LOWER_LIMIT_CONSTRAINT * 60));
				$upperLimit = (time() + (SCHEDULE_UPPER_LIMIT_CONSTRAINT * 60));
				if($ticketScheduleTime < $lowerLimit || $ticketScheduleTime > $upperLimit){
					$this->set('success', "FALSE");
					$this->set('message', "Ticket verification failed due to time constraints!");

				} else {
					// everything is OK
					$tickets = array("ticket" => array());
					$tickets['ticket'][] = $this->getTicketFromResultSet($ticket, $ticketsObjModel);
					$data = array('tickets' => $tickets);

					$this->set('data', $data);
				}
			}

		} else {
			$this->set('success', "FALSE");
			$this->set('message', "Ticket code not recognized!");
		}
	}

	/**
	 * get a scheduled ticket
	 * @param ticket_code: ticket_number of the scheduled ticket
	 **/
	function get(){
		$ticket_code = $this->request->params['ticket_code'];

		require_once("services/MethodBase.class.php");
		require_once("services/contactline/Tickets.php");
		$ticketsObjModel = new Tickets();
		
		$ticketsResultSet = $ticketsObjModel->getTicketByCode($ticket_code);

		if(count($ticketsResultSet) > 0){
			$tickets = array("ticket" => array());
			$tickets['ticket'][] = $this->getTicketFromResultSet($ticketsResultSet[0], $ticketsObjModel);

			$data = array('tickets' => $tickets);

			$this->set('success', "TRUE");
			$this->set('data', $data);

		} else {
			$this->set('success', "FALSE");
			$this->set('message', "ticket code not recognized!");
		}
	}

	private function getTicketFromResultSet($ticketResultSet, $ticketObjModel){

		$ticket = array(
			"id" => $ticketResultSet['senhas_id_senha'], 
			"ticket_type" => "normal", // "schedule",
			"service_id" => $ticketResultSet['senhas_servico_destino'], // "1",
			"service" => $ticketResultSet['servico_destino_nome'], // "Serv1",
			"ticket_code" => "", // "610302", // only if is schedule ticket
			"ticket_number" => "", // "A25", // only if is a normal ticket
			"scheduled" => $ticketResultSet['senhas_scheduled'], // "2012-04-17 12:10:00",
			"status" => $ticketResultSet['senhas_status'], // "SCHEDULED",
			"client_name" => $ticketResultSet['senhas_nome_cliente'], // "José Meireles",
			"comments" => $ticketResultSet['senhas_observacoes'], // "Agendamento para serviço de Serv1; Necessidade de acesso para deficientes;",
		);

		// fix ticket_type and ticket_number/code
		switch($ticketResultSet['servico_emissao_tipo_servico']){
		case 'agendamentos':
			$ticket['ticket_type'] = "schedule";
			$ticket['ticket_code'] = $ticketResultSet['senhas_numero_senha'];
			break;

		default:
			$ticket['ticket_type'] = "normal";
			$ticket['ticket_code'] = $ticketResultSet['servico_emissao_identificador'] . $ticketResultSet['senhas_numero_senha'];
			break;
		}

		return $ticket;
	}
}
?>
